/* gpsoundbuf.h - Implements a ring buffer for sound mixing on the GamePark 32
 *
 * Copyright (c)2002 Christian Nowak <chnowak@web.de>
 *
 * Use it as you want without any obligations or restrictions,
 * but a notification via eMail would be nice.
 *
 */

#ifndef __GPSOUNDBUF_H__
#define __GPSOUNDBUF_H__

/* Header files */
#include "defines.h"
#include "gpmm.h"
#include "gpmem.h"
#include "gpos_def.h"

typedef struct GPSOUNDBUF
  {
    PCM_SR freq;              /* Taken from gpmm.h */
    PCM_BIT format;           /* Taken from gpmm.h */
    unsigned int samples;     /* Buffer length (in samples) */
    void * userdata;          /* Userdata which gets passed to the callback function */
    void (*callback)(         /* Callback function (just like in SDL) */
          void * userdata,    /* GPSOUNDBUF.userdata */
          u8 * stream,        /* Pointer to the buffer which needs to be refilled */
          int len);           /* Length of the buffer in bytes */
    unsigned int pollfreq;    /* Frequency of the timer interrupt which polls the playing position
                               * recommended value: 2*(playingfreq in Hz/GPSOUNDBUF.samples) */
    unsigned int samplesize;  /* Size of one sample (8bit mono->1, 16bit stereo->4) - don't touch this */
  } GPSOUNDBUF;

/* Global variables */
unsigned int frame;
unsigned int * soundPos;
volatile int idx_buf;
unsigned int shiftVal;
void * buffer;
GPSOUNDBUF soundBuf;

int GpSoundBufStart ( GPSOUNDBUF * );
void GpSoundBufStop ( void );

/* This routine gets called by the timer interrupt and
 * polls the current playing position within the buffer.
 */
void soundtimer ( void )
  {
    extern int callback_count;
    extern int callback_count_ok;
    unsigned int t = (((unsigned int)(*soundPos) - (unsigned int)buffer)>>shiftVal) >= soundBuf.samples ? 1 : 0;
    if (t!=frame)
    {
        /* Sound updated ? */
        if (callback_count!=callback_count_ok) return;

        unsigned int offs = ((frame==1) ? (soundBuf.samples<<shiftVal) : 0);
        soundBuf.callback(soundBuf.userdata, (u8*)((unsigned int)buffer+offs), soundBuf.samples<<shiftVal);
        frame = t;

        /* Sound updated ! */
        callback_count++;
    }
    else
    {
        callback_count = 0;
	callback_count_ok = 0;
    }
  }

int GpSoundBufStart ( GPSOUNDBUF * sb )
  {
    frame = 0;

    /* Copy the structure */
    asm_memcpy ( (GPSOUNDBUF *)&soundBuf, sb, sizeof(GPSOUNDBUF) );

    /* Calculate size of a single sample in bytes
     * and a corresponding shift value
     */
    shiftVal = 0;
    switch (soundBuf.freq)
      {
        case PCM_S11:
        case PCM_S22:
        case PCM_S44:
          shiftVal++;
          break;
		default: break;
      }
    if (soundBuf.format == PCM_16BIT)
      shiftVal++;
    soundBuf.samplesize = 1<<shiftVal;

    /* Allocate memory for the playing buffer */
    buffer = (void *)gm_malloc ( soundBuf.samplesize*soundBuf.samples*2 );
    asm_memset ( buffer, 0x80, soundBuf.samplesize*soundBuf.samples*2 );
    swi_mmu_change((int)buffer & 0xfffff000, ((int)buffer & 0xfffff000)+4095, 0xFFA ); /*no cache for samplebuffer*/

    /*inicia el Sonido*/
    GpPcmInit(soundBuf.freq,soundBuf.format);
	
    /* Set timer interrupt #0 */
    /*GpTimerOptSet ( 0, soundBuf.pollfreq, 0, soundtimer );
    GpTimerSet ( 0 );*/
    if ( GPOS_ERR_ALREADY_USED ==
	 GpTimerOptSet ( 0, soundBuf.pollfreq, 0, soundtimer ) )
    {
      GpTimerKill(0);
    }
    GpTimerSet ( 0 );
    
    /* Start playing */
    GpPcmPlay ( (unsigned short*)buffer, soundBuf.samples*soundBuf.samplesize*2, 1 );
    GpPcmLock ( (unsigned short*)buffer, (int*)&idx_buf, (unsigned int*)&soundPos );

    return 0;
  }


void GpSoundBufStop ( void )
  {
    GpPcmStop();
    GpPcmRemove ( (unsigned short*)buffer );
    GpTimerKill ( 0 );
    gm_free ((void *)buffer);
  }

#endif
